package edu.cornell.cs.cs4120.eth.etac;

import edu.cornell.cs.cs4120.eth.SourceFileTest;
import edu.cornell.cs.cs4120.eth.SourceFileTestCollection;
import edu.cornell.cs.cs4120.eth.etac.tester.EtacOptTester;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

public class EtacOptTestDriver extends EtacTestDriver {

    public static final Set<String> opts;

    protected static final int TIMEOUT = 30;
    protected static final TimeUnit TIMEUNIT = TimeUnit.SECONDS;

    static {
        opts = new HashSet<>();
        opts.add("");
        opts.add("cf");
        opts.add("reg");
        opts.add("mc");
        opts.add("uce");
        opts.add("cse");
        opts.add("alg");
        opts.add("copy");
        opts.add("dce");
        opts.add("inl");
        opts.add("sr");
        opts.add("lu");
        opts.add("licm");
        opts.add("pre");
        opts.add("cp");
        opts.add("vn");
    }

    protected String opt;
    protected boolean optImplemented;

    public EtacOptTestDriver(SourceFileTestCollection sftc, String opt) {
        super(sftc);
        if (opt != null && !opts.contains(opt))
            throw new UnsupportedOperationException("Optimization flag not supported: " + opt);
        this.opt = opt;
    }

    @Override
    public boolean shouldExecute() {
        return opt == null || opt.isEmpty() || optImplemented;
    }

    public boolean optImplemented() {
        return optImplemented;
    }

    @Override
    public void addCodeGenTest() {
        testers.add(new EtacOptTester(opt));
    }

    @Override
    public boolean preTest(SourceFileTestCollection sftc) {
        // Check if the compiler implements this optimization.
        String compilerDirname = getPathFromFlagMap("compilerpath");
        String compilerCmd = "";
        File compilerDir = new File(compilerDirname);
        if (!compilerDir.isAbsolute()) {
            Path wd = Paths.get(".").toAbsolutePath();
            compilerCmd += wd.toString() + File.separator;
        }
        compilerCmd += compilerDirname + commandName();
        // Check existence of compiler command
        File compiler = new File(compilerCmd);
        if (!compiler.isFile()) {
            sftc.appendFailureMessage("File " + compilerCmd + " does not exist.");
            return false;
        }
        List<String> cmdLine = new ArrayList<>(2);
        cmdLine.add(compilerCmd);
        cmdLine.add("--report-opts");
        ProcessBuilder pb = new ProcessBuilder(cmdLine);
        // Invoke compiler.
        try {
            Process p = pb.start();
            if (!p.waitFor(TIMEOUT, TIMEUNIT)) p.destroyForcibly();
            InputStream outStream = p.getInputStream(); // normal output of the shell
            InputStream errStream = p.getErrorStream(); // error output of the shell
            try (InputStreamReader or = new InputStreamReader(outStream);
                    InputStreamReader er = new InputStreamReader(errStream);
                    BufferedReader osr = new BufferedReader(or);
                    BufferedReader esr = new BufferedReader(er)) {
                String line = null;
                List<String> opts = new LinkedList<>();
                while ((line = osr.readLine()) != null) opts.add(line.trim());
                optImplemented = opts.contains(opt);
            }
            return true;
        } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return false;
        } catch (IOException e) {
            sftc.appendFailureMessage(e.getMessage());
            return false;
        }
    }

    @Override
    public boolean preTest(SourceFileTest t) {
        // Run the compiler with optimization off
        String compilerDirname = getPathFromFlagMap("compilerpath");
        List<String> cmdLineHdr = new LinkedList<>(t.getCommandLineHeader());
        if (opt != null) {
            if (!opt.isEmpty()) cmdLineHdr.remove("-O" + opt);
            if (opt.isEmpty() || opt.equals("reg")) cmdLineHdr.add(0, "-O");
        }
        for (List<String> list : t.getSourceFileNames()) {
            List<String> cmdLine = new LinkedList<>(cmdLineHdr);
            cmdLine.addAll(list);

            // Invoke the compiler on the compilation unit.
            int ret = t.invokeCompiler(compilerDirname, cmdLine);
            if (ret != 0) {
                t.appendFailureMessage(
                        "Failed to compile: " + t.compilerName() + " exit code " + ret);
                return false;
            }
        }

        boolean okay = super.preTest(t);
        if (!okay) return false;
        return true;
    }

    @Override
    public void getSummary(StringBuffer sb) {
        if (opt == null) return;
        if (opt != null && !optImplemented) {
            sb.append("\nOptimization " + opt + " unimplemented");
            return;
        }
        super.getSummary(sb);
    }
}
